#ifndef _MODULE_BLOWFISH_H_
#define _MODULE_BLOWFISH_H_
/*
 *              blowfish - Blowfish Library
 *
 *    Copyright (c) 1998-2015 by CryptoSoft GmbH. All Rights Reserved.
 *              http://www.cryptosoft.de, info@cryptosoft.de
 *
 *    Synopsis:	blowfish_init(key,len,pks)
 * Description: intializes all arrays and permutation tables for bf
 *       Input: pks  -  pointer to key schedule array
 *              key  -  Blowfish key
 *              len  -  length of key
 *      Output: 0 if OK; >0 if a (semi) weak was selected
 *
 *    Synopsis: blowfish_ecbYcode(in64,out64,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) 64-bit in to
 *              64-bit out using Blowfish in ECB mode
 *     	 Input: in64  - pointer to 64-bit buffer of input data
 *              out64 - pointer to 64-bit buffer for output data
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: blowfish_cbcYcode(in64,out64,iv64,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) 64-bit in to
 *              64-bit out using Blowfish in CBC mode
 *     	 Input: in64  - pointer to 64-bit buffer of input data
 *              out64 - pointer to 64-bit buffer for output data
 *              iv64  - pointer to 64-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: blowfish_cfbYcode(inpb,outpb,len,iv64,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using Blowfish in 8-bit CFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv64  - pointer to 64-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: blowfish_ofbYcode(inpb,outpb,len,iv64,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using Blowfish in 8-bit OFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv64  - pointer to 64-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 */


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#define ROUNDS 16

typedef struct blowfish_schedule
{
  unsigned long P[ROUNDS + 2];
  unsigned long S[4*256];
} BLOWFISH_KS;

#if defined(WIN16)
#define DECL int far pascal
#define UCHAR unsigned char far
#define ULONG unsigned long far
#define SIZE_T size_t far
#define BLOWFISH_PKS BLOWFISH_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define DECL int pascal
#define UCHAR unsigned char
#define ULONG unsigned long
#define SIZE_T size_t
#define BLOWFISH_PKS BLOWFISH_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define DECL int
#define UCHAR unsigned char
#define ULONG unsigned long
#define SIZE_T size_t
#define BLOWFISH_PKS BLOWFISH_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL blowfish_init(UCHAR *key, int len, BLOWFISH_PKS pks);
DECL blowfish_ecbencode(UCHAR *in64, UCHAR *out64, BLOWFISH_PKS pks);
DECL blowfish_ecbdecode(UCHAR *in64, UCHAR *out64, BLOWFISH_PKS pks);
DECL blowfish_cbcencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_cbcdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, BLOWFISH_PKS pks);

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* _MODULE_BLOWFISH_H_ */
