#ifndef _MODULE_CAST128_H_
#define _MODULE_CAST128_H_
/*
 *             cast128 - Cast-128 Library
 *
 *    Copyright (c) 1998-2014 by CryptoSoft GmbH. All Rights Reserved.
 *             http://www.cryptosoft.de, info@cryptosoft.de
 *
 *    Synopsis:	cast128_init(key,len,pks)
 * Description: intializes all arrays and permutation tables for cast128
 *       Input: pks  -  pointer to key schedule array
 *              key  -  Cast-128 key
 *              len  -  length of key in byte (max 16)
 *      Output: 0 if OK; >0 if a (semi) weak was selected
 *
 *    Synopsis: cast128_ecbYcode(in64,out64,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) 64-bit in to
 *              64-bit out using Cast-128 in ECB mode
 *     	 Input: in64  - pointer to 64-bit buffer of input data
 *              out64 - pointer to 64-bit buffer for output data
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: cast128_cbcYcode(in64,out64,iv64,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) 64-bit in to
 *              64-bit out using Cast-128 in CBC mode
 *     	 Input: in64  - pointer to 64-bit buffer of input data
 *              out64 - pointer to 64-bit buffer for output data
 *              iv64  - pointer to 64-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: cast128_cfbYcode(inpb,outpb,len,iv64,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using Cast-128 in 8-bit CFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv64  - pointer to 64-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: cast128_ofbYcode(inpb,outpb,len,iv64,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using Cast-128 in 8-bit OFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv64  - pointer to 64-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *
 */


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef struct cast128_schedule {
  unsigned long ks[32];
} CAST128_KS;

#if defined(WIN16)
#define DECL int far pascal
#define UCHAR unsigned char far
#define ULONG unsigned long far
#define SIZE_T size_t far
#define CAST128_PKS CAST128_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define DECL int pascal
#define UCHAR unsigned char
#define ULONG unsigned long
#define SIZE_T size_t
#define CAST128_PKS CAST128_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define DECL int
#define UCHAR unsigned char
#define ULONG unsigned long
#define SIZE_T size_t
#define CAST128_PKS CAST128_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */


DECL cast128_init(UCHAR *key16, int len, CAST128_PKS pks);
DECL cast128_ecbencode(UCHAR *in64, UCHAR *out64, CAST128_PKS pks);
DECL cast128_ecbdecode(UCHAR *in64, UCHAR *out64, CAST128_PKS pks);
DECL cast128_cbcencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_cbcdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, CAST128_PKS pks);

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* _MODULE_CAST128_H_ */
