/*
 *  Copyright (c) by CryptoSoft GmbH 1998-2014
 *  All Rights Reserved
 *  Licensed Material - Property of CryptoSoft GmbH
 *  This software is made available solely pursuant to the
 *  terms of a license agreement which governs its use.
 *
 *  This file encryption sample source will give you an 
 *  idea on how to call up the various library functions.
 *
 *  Usage: cipher [-d] [-m <mode>] [-k <key>] [-q] infile outfile
 *    -d        : decrypt
 *    -m <mode> : operation mode (mode := ecb | cbc | ofb | cfb)
 *    -k <key>  : key used for encryption/decryption
 *    -q        : quiet operation
 *    infile    : name of file to process
 *    outfile   : name of output file
 */

#ifdef WIN32
#include <windows.h>
#endif

#ifdef WIN64
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include "cast256.h"

#define BUFSIZE 16384

#define KEYSZ  (32)
#define BLKSZ  (16)

#define EMODE_ECB 0   /* electronic code book mode  */
#define EMODE_CBC 1   /* cipher block chaining mode */
#define EMODE_OFB 2   /* 8 bit output feedback mode */
#define EMODE_CFB 3   /* 8 bit cipher feedback mode */

static char *algo  = "Cast-256";

static char *usage = "[-d] [-m <mode>] [-k <key>] [-q] infile outfile";

static CAST256_KS ks;

int main(int argc, char **argv)
{
  register unsigned char *buf, *prog, *b;
  static unsigned char key[KEYSZ]  = {0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF };
  static unsigned char ivec[BLKSZ] = {0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD, 0xEF }, iv[BLKSZ];
  register int i, bytes, mode, quiet, decrypt, found_problem;
  static int l;
  static unsigned char lb;
  static long fsize;
  register time_t t, t1;
  FILE *ih, *oh;

  quiet = decrypt = found_problem = l = 0;

  mode  = EMODE_CBC;  /* default encryption mode */

  prog  = *argv;
  argc--, argv++;

  while (argc > 0) {
    if (strcmp(*argv, "-d") == 0) {
      decrypt = 1;
    } else
    if (strcmp(*argv, "-m") == 0) {
      argc--; argv++;
      if (argc) {
        if (strcmp(*argv, "ecb") == 0) mode = EMODE_ECB; else
        if (strcmp(*argv, "cbc") == 0) mode = EMODE_CBC; else 
        if (strcmp(*argv, "ofb") == 0) mode = EMODE_OFB; else
        if (strcmp(*argv, "cfb") == 0) mode = EMODE_CFB; else
        found_problem = 1;
      }
    } else
    if (strcmp(*argv, "-k") == 0) {
      argc--; argv++;
      if (argc) {
        memset(key, 0, KEYSZ);
        bytes = strlen(*argv);
        memcpy(key, *argv, bytes > KEYSZ - 1 ? KEYSZ:bytes);
      }
    } else
    if (strcmp(*argv, "-q") == 0) {
      quiet = 1;
    } else break;
    argc--, argv++;
  }

  if (argc != 2 || found_problem) {
    fprintf(stderr,"Usage : %s %s \n", prog, usage);
    return 1;
  }

  if ((buf = (unsigned char *) malloc(BUFSIZE)) == NULL) {
    fprintf(stderr,"%s: Error: Cannot alloc memory !\n", prog);
    return 1;
  }

  ih = fopen(*argv++,"rb");  /* open input file for reading */

  oh = fopen(*argv,"wb");    /* create output file */

  if (ih == NULL || oh == NULL) {
    perror(ih ? *argv : *--argv);
    return 1;
  }

  /* get input file size */
  fseek(ih, 0L, SEEK_END);
  fsize = ftell(ih);
  fseek(ih, 0L, SEEK_SET);

  if (decrypt && (mode == EMODE_ECB || mode == EMODE_CBC)) {
    if (fsize % BUFSIZE == 1) found_problem = 1; else found_problem = 0;
    if (found_problem) {
      fseek(ih, fsize - 1, SEEK_SET);
      /* read & save the last byte */
      fread(&lb, 1, 1, ih);
      fseek(ih, 0L, SEEK_SET);
    }
  }

  /* initialize key schedule array */
  cast256_init(key, KEYSZ, &ks);

  if (mode == EMODE_CBC || mode == EMODE_OFB || mode == EMODE_CFB) memcpy(iv, ivec, 8);

  if (!quiet) t = time(NULL);

  while ((bytes = fread(buf, 1, BUFSIZE, ih)) != 0) {
    if (bytes == 1 && found_problem) break;
    /* block modes */
    if (mode == EMODE_ECB || mode == EMODE_CBC) {
      for (b = buf, i = bytes; bytes > 0; bytes -= BLKSZ , b += BLKSZ) {
        if (!decrypt) {
          if (bytes < BLKSZ) {
            i += BLKSZ - bytes;
            l = bytes;
          }
          if (mode == EMODE_ECB) {
            cast256_ecbencode(b, b, &ks);
          } else {
            cast256_cbcencode(b, b, iv, &ks);
            memcpy(iv, b, 8);
          }
        } else {
          if (bytes > BLKSZ - 1) {
            if (mode == EMODE_ECB) {
              cast256_ecbdecode(b, b, &ks);
            } else {
              memcpy(iv, b, 8);
              cast256_cbcdecode(b, b, ivec, &ks);
              memcpy(ivec, iv, 8);
            }
          } else i -= (BLKSZ + 1 - (b[0] & 0xff));
        }
      }
      bytes = i;
      if (decrypt && found_problem && (bytes + 1 == fsize)) bytes = bytes - BLKSZ + lb;
    }
    /* 8 bit feedback modes */
    if (mode == EMODE_OFB || mode == EMODE_CFB) {
      if (!decrypt) {
        if (mode == EMODE_OFB)
        cast256_ofbencode(buf, buf, bytes, iv, &ks);
        else
        cast256_cfbencode(buf, buf, bytes, iv, &ks);
      } else {
        if (mode == EMODE_OFB)
        cast256_ofbdecode(buf, buf, bytes, iv, &ks);
        else
        cast256_cfbdecode(buf, buf, bytes, iv, &ks);
      }
    }
    if (fwrite(buf, 1, bytes, oh) != (size_t) bytes) {
      fprintf(stderr,"%s: Error in write !\n", prog);
      return 1;
    }
    fsize -= bytes;
  }

  if (!decrypt && mode < EMODE_OFB) fputc(l ? l : l + BLKSZ, oh);

  if (!quiet) {
    t1 = time(NULL) - t;
    printf("Processed bytes per second using %s: %ld\n", algo, ftell(ih)/(t1 ? t1 : 1));
  }

  fclose(ih);
  fclose(oh);
  return 0;
}
