/*
 *  Copyright (c) by CryptoSoft GmbH 1998-2017
 *  All Rights Reserved
 *  Licensed Material - Property of CryptoSoft GmbH
 *  This software is made available solely pursuant to the
 *  terms of a license agreement which governs its use.
 *
 *  Although there is no official standard for doing DES3-EDE
 *  the library functions use a 16 byte key to perform this
 *  mode in a way that has been adopted by the American Bankers 
 *  Association (described in ANSI X9.17, Financial Institution 
 *  Key Management, 1985). In order to do so - to get the cipher 
 *  text we do a C = Ek1(Dk2(Ek1(P))), to get the plain text we 
 *  do the reverse P = Dk1(Ek2(Dk1(C))). 
 *
 *  However, some clients requested a version that is compatible
 *  to several other third party implementations which use a 24 byte
 *  key for this mode. This file sample source will give you an idea 
 *  on how to perform 3DES in ECB or CBC mode and EDE order using a 
 *  24 byte key with the existing library functions.  
 *
 *  Usage: 3des [-d] [-k <key>] [-m <mode>] [-v <vector>] infile outfile
 *    -d          : decrypt
 *    -m <mode>   : encryption mode (mode := ecb | cbc)
 *    -k <key>    : key used for encryption/decryption
 *    -v <vector> : initialization vector 
 *    infile      : name of file to process
 *    outfile     : name of output file
 */

#ifdef WIN32
#include <windows.h>
#endif

#ifdef WIN64
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include "des3.h"

#define BUFSIZE 16384
#define KEYSIZE 24
#define BLKSIZE 8
#define EMODE_ECB 0
#define EMODE_CBC 1

static char *algo_name = "3DES-EDE"; 

static char *usage = "[-d] [-m <mode>] [-k <key>] [-v <vector>] infile outfile";

void xor8(unsigned char a64[8], unsigned char b64[8], unsigned char out64[8])
{
  out64[0] = a64[0] ^ b64[0]; out64[1] = a64[1] ^ b64[1];
  out64[2] = a64[2] ^ b64[2]; out64[3] = a64[3] ^ b64[3];
  out64[4] = a64[4] ^ b64[4]; out64[5] = a64[5] ^ b64[5];
  out64[6] = a64[6] ^ b64[6]; out64[7] = a64[7] ^ b64[7];
}
 
int des_3EDEinit_3keys(unsigned char key[24], int len, DES_KS k1, DES_KS k2, DES_KS k3)
{
  int ret;
  ret  = des_init(key, 8, k1);
  ret |= des_init(&key[8], 8, k2);
  ret |= des_init(&key[16], 8, k3);
  return ret;
}
 
void des_ecb3EDEencode_3keys(unsigned char *in64, unsigned char *out64, DES_KS k1, DES_KS k2, DES_KS k3)
{
  unsigned char t64[8], s64[8];
  des_ecbencode(in64, t64, k1);
  des_ecbdecode(t64, s64, k2);
  des_ecbencode(s64, out64, k3);
}

void des_ecb3EDEdecode_3keys(unsigned char *in64, unsigned char *out64, DES_KS k1, DES_KS k2, DES_KS k3)
{
  unsigned char t64[8], s64[8];
  des_ecbdecode(in64, t64, k1);
  des_ecbencode(t64, s64, k2);
  des_ecbdecode(s64, out64, k3);
}

void des_cbc3EDEencode_3keys(unsigned char *in64, unsigned char *out64, unsigned char *ivec, DES_KS k1, DES_KS k2, DES_KS k3) 
{ 
  unsigned char t64[8], s64[8]; 
  xor8(in64, ivec, t64); 
  des_ecbencode(t64, s64, k1); 
  des_ecbdecode(s64, t64, k2); 
  des_ecbencode(t64, out64, k3); 
} 

void des_cbc3EDEdecode_3keys(unsigned char *in64, unsigned char *out64, unsigned char *ivec, DES_KS k1, DES_KS k2, DES_KS k3) 
{ 
  unsigned char t64[8], s64[8]; 
  des_ecbdecode(in64, t64, k1); 
  des_ecbencode(t64,  s64, k2); 
  des_ecbdecode(s64,  t64, k3); 
  xor8(t64, ivec, out64); 
}

int encrypt_file(char *fni, char *fno, unsigned char *key, unsigned char *ivec, int mode)
{
  DES_KS ks1, ks2, ks3;
  int lastb;
  FILE *ih, *oh;
  unsigned char *buf, *b, iv[BLKSIZE];
  int bytes, cnt;
  

  ih = fopen(fni,"rb");  /* open input file for reading */
  oh = fopen(fno,"wb");  /* create output file */

  if (ih == NULL || oh == NULL) {
    perror(ih ? fno : fni);
    return 1;
  }
  
  /* allocate buffer */
  if ((buf = (unsigned char *) malloc(BUFSIZE)) == NULL) {
    fprintf(stderr,"Error: Cannot alloc memory !\n");
    return 1;
  }

  /* initialize key schedule array */
  des_3EDEinit_3keys(key, KEYSIZE, ks1, ks2, ks3);
  
  /* copy initialization vector */
  if (mode == EMODE_CBC) memcpy(iv, ivec, BLKSIZE);

  /* process the input file data */
  while ((bytes = (int) fread(buf, 1, BUFSIZE, ih)) != 0) {
    for (b = buf, cnt = bytes; bytes > 0; bytes -= BLKSIZE , b += BLKSIZE) {
      if (bytes < BLKSIZE) {
        cnt += BLKSIZE - bytes;
        lastb = bytes;
      }
      if (mode == EMODE_CBC) {
        des_cbc3EDEencode_3keys(b, b, iv, ks1, ks2, ks3);
        memcpy(iv, b, 8);
      } else {
        des_ecb3EDEencode_3keys(b, b, ks1, ks2, ks3);
      }
    }
    bytes = cnt;
    if (fwrite(buf, 1, bytes, oh) != (size_t) bytes) {
      fprintf(stderr,"Error in write !\n");
      return 1;
    }
  }

  /* how many valid bytes in last output block */
  fputc(lastb ? lastb : lastb + BLKSIZE, oh);

  fclose(ih);
  fclose(oh);
  return 0;
}

int decrypt_file(char *fni, char *fno, unsigned char *key, unsigned char *ivec, int mode)
{
  DES_KS ks1, ks2, ks3;
  FILE *ih, *oh;
  unsigned char *buf, *b, iv[BLKSIZE], lb;
  int bytes, cnt, fp;
  long fsize;

  ih = fopen(fni,"rb");  /* open input file for reading */
  oh = fopen(fno,"wb");  /* create output file */

  if (ih == NULL || oh == NULL) {
    perror(ih ? fno : fni);
    return 1;
  }
  
  /* get input file size */
  fseek(ih, 0L, SEEK_END);
  fsize = ftell(ih);
  fseek(ih, 0L, SEEK_SET);

  /* check for the case if BUFSIZE eq FSIZE + 1 */
  if (fsize % BUFSIZE == 1) fp = 1; else fp = 0;
  if (fp) {
    fseek(ih, fsize - 1, SEEK_SET);
    /* read & save the last byte */
    fread(&lb, 1, 1, ih);
    fseek(ih, 0L, SEEK_SET);
  }

  /* allocate buffer */
  if ((buf = (unsigned char *) malloc(BUFSIZE)) == NULL) {
    fprintf(stderr,"Error: Cannot alloc memory !\n");
    return 1;
  }

  /* initialize key schedule array */
  des_3EDEinit_3keys(key, KEYSIZE, ks1, ks2, ks3);
  
  /* copy initialization vector */
  if (mode == EMODE_CBC) memcpy(iv, ivec, BLKSIZE);

  /* process the input file data */
  while ((bytes = (int) fread(buf, 1, BUFSIZE, ih)) != 0) {
    if (bytes == 1 && fp) break;
    for (b = buf, cnt = bytes; bytes > 0; bytes -= BLKSIZE , b += BLKSIZE) {
      if (bytes > BLKSIZE - 1) {
        if (mode == EMODE_CBC) {
          memcpy(iv, b, BLKSIZE);
          des_cbc3EDEdecode_3keys(b, b, ivec, ks1, ks2, ks3);
          memcpy(ivec, iv, BLKSIZE);
        } else {
          des_ecb3EDEdecode_3keys(b, b, ks1, ks2, ks3);
        }
      } else cnt -= (BLKSIZE + 1 - (b[0] & 0xff));
    }
    bytes = cnt;
    if (fp && (bytes + 1 == fsize)) bytes = bytes - BLKSIZE + lb;
    if (fwrite(buf, 1, bytes, oh) != (size_t) bytes) {
      fprintf(stderr,"Error in write !\n");
      return 1;
    }
  }

  fclose(ih);
  fclose(oh);
  free(buf);
  return 0;
}

int main(int argc, char **argv)
{
  unsigned char key[KEYSIZE]  = {0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD, 0xEF, 0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD, 0xEF, 0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD, 0xEF };
  unsigned char ivec[BLKSIZE] = {0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD, 0xEF }, *prog;
  char *fIn, *fOut;
  register int decrypt, bytes, mode;

  decrypt = 0;
  mode    = EMODE_ECB;

  prog  = *argv;
  argc--, argv++;

  while (argc > 0) {
    if (strcmp(*argv, "-d") == 0) {
      decrypt = 1;
    } else 
    if (strcmp(*argv, "-m") == 0) {
      argc--; argv++;
      if (argc) {
        if (strcmp(*argv, "cbc") == 0) mode = EMODE_CBC; else
        if (strcmp(*argv, "ecb") == 0) mode = EMODE_ECB; 
      }
    } else
    if (strcmp(*argv, "-k") == 0) {
      argc--; argv++;
      if (argc) {
        memset(key, 0, KEYSIZE);
        bytes = (int) strlen(*argv);
        memcpy(key, *argv, bytes > KEYSIZE - 1 ? KEYSIZE:bytes);
      }
    } else
    if (strcmp(*argv, "-v") == 0) {
      argc--; argv++;
      if (argc) {
        memset(ivec, 0, BLKSIZE);
        bytes = (int) strlen(*argv);
        memcpy(ivec, *argv, bytes > BLKSIZE - 1 ? BLKSIZE:bytes);
      }
    } else break;
    argc--, argv++;
  }

  if (argc != 2) {
    fprintf(stderr,"Usage : %s %s \n", prog, usage);
    return 1;
  }
   
  fIn  = *argv++;
  fOut = *argv;  

  return decrypt ? decrypt_file(fIn, fOut, key, ivec, mode) : encrypt_file(fIn, fOut, key, ivec, mode);  
}
