#ifndef _MODULE_DES3_H_
#define _MODULE_DES3_H_
/*
 *             des3 - NBS Data Encryption Standard Library
 *
 *    Copyright (c) 1998-2017 by CryptoSoft GmbH. All Rights Reserved.
 *             http://www.cryptosoft.de, info@cryptosoft.de
 *
 *    Synopsis:	des_init(key, len,  ks)
 * Description: intializes all arrays and permutation tables for
 *              single DES
 *       Input: key -  DES key  (8 byte max)
                len -  length of key
 *              ks  -  key schedule
 *      Output: 0 if OK; >0 if a (semi) weak was selected
 *
 *    Synopsis: des_ecbXcode(in64, out64, ks)
 * Description: encrypts (X=en) or decrypts (X=de) 64-bit inblock to
 *              64-bit outblock using single DES in ECB mode
 *     	 Input: in64  - pointer to 64-bit buffer of input data
 *              out64 - pointer to 64-bit buffer for output data
 *              ks    - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis: des_cbcXcode(in64, out64, iv64, ks)
 * Description: encrypts (X=en) or decrypts (X=de) 64-bit inblock to
 *              64-bit outblock using single DES in CBC mode
 *     	 Input: in64  - pointer to 64-bit buffer of input data
 *              out64 - pointer to 64-bit buffer for output data
 *              iv64   - initalization vector
 *              ks    - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis: des_cfbYcode(inpb,outpb,len,iv64,ks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using DES in 8-bit CFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv64  - pointer to 64-bit initialization vector
 *              ks    - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis: des_ofbYcode(inpb,outpb,len,iv64,ks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using DES in 8-bit OFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv64  - pointer to 64-bit initialization vector
 *              ks    - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis:	des_3EDEinit(key, len, ks1, ks2)
 * Description: intializes all arrays and permutation tables for
 *              triple DES in EDE (encrypt-decrypt-encrypt) mode
 *       Input: key -  3DES key (16 byte max)
 *              ks1 -  key schedule
 *              ks2 -  key schedule
 *      Output: 0 if OK; >0 if a (semi) weak was selected
 *
 *    Synopsis: des_ecb3EDEXcode(in64, out64, ks1, ks2)
 * Description: encrypts (X=en) or decrypts (X=de) 64-bit inblock to
 *              64-bit outblock using triple DES in ECB-EDE mode
 *     	 Input: in64  - pointer to 64-bit buffer of input data
 *              out64 - pointer to 64-bit buffer for output data
 *              ks1   - key schedule
 *              ks2   - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis: des_cbc3EDEXcode(in64, out64, iv64, ks1, ks2)
 * Description: encrypts (X=en) or decrypts (X=de) 64-bit inblock to
 *              64-bit outblock using triple DES in CBC-EDE mode
 *     	 Input: in64  - pointer to 64-bit buffer of input data
 *              out64 - pointer to 64-bit buffer for output data
 *              iv64   - initalization vector
 *              ks1   - key schedule
 *              ks2   - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis: des_cfb3EDEXcode(inpb, outpb, len, iv64, ks1, ks2)
 * Description: encrypts (X=en) or decrypts (X=de) a given data block
 *              using triple DES in 8-bit CFB-EDE mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv64  - pointer to 64-bit initialization vector
 *              ks1   - key schedule
 *              ks2   - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis: des_ofb3EDEXcode(inpb, outpb, len, iv64, ks1, ks2)
 * Description: encrypts (X=en) or decrypts (X=de) a given data block
 *              using triple DES in 8- bit OFB-EDE mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv64  - pointer to 64-bit initialization vector
 *              ks1   - key schedule
 *              ks2   - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis:	des_3EEEinit(key, int len, ks1, ks2, ks3)
 * Description: intializes all arrays and permutation tables for
 *              triple DES in EEE (encrypt-encrypt-encrypt) mode
 *       Input: key -  3DES key (24 byte max)
 *              ks1 -  key schedule
 *              ks2 -  key schedule
 *              ks3 -  key schedule
 *      Output: 0 if OK; >0 if a (semi) weak was selected
 *
 *    Synopsis: des_ecb3EEEXcode(in64, out64, ks1, ks2, ks3)
 * Description: encrypts (X=en) or decrypts (X=de) 64-bit inblock to
 *              64-bit outblock using triple DES in ECB-EEE mode
 *     	 Input: in64  - pointer to 64-bit buffer of input data
 *              out64 - pointer to 64-bit buffer for output data
 *              ks1   - key schedule
 *              ks2   - key schedule
 *              ks3   - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis: des_cbc3EEEXcode(in64, out64, iv64, ks1, ks2, ks3)
 * Description: encrypts (X=en) or decrypts (X=de) 64-bit inblock to
 *              64-bit outblock using triple DES in CBC-EEE mode
 *     	 Input: in64  - pointer to 64-bit buffer of input data
 *              out64 - pointer to 64-bit buffer for output data
 *              iv64  - initalization vector
 *              ks1   - key schedule
 *              ks2   - key schedule
 *              ks3   - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis: des_cfb3EEEXcode(inpb, outpb, len, iv64, ks1, ks2, ks3)
 * Description: encrypts (X=en) or decrypts (X=de) a given data block
 *              using triple DES in 8-bit CFB-EEE mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv64  - pointer to 64-bit initialization vector
 *              ks1   - key schedule
 *              ks2   - key schedule
 *              ks3   - key schedule
 *      Output: 0 if OK
 *
 *    Synopsis: des_ofb3EEEXcode(inpb, outpb, len, iv64, ks1, ks2, ks3)
 * Description: encrypts (X=en) or decrypts (X=de) a given data block
 *              using triple DES in 8-bit OFB-EEE mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv64  - pointer to 64-bit initialization vector
 *              ks1   - key schedule
 *              ks2   - key schedule
 *              ks3   - key schedule
 *      Output: 0 if OK
 */


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifdef X64
typedef unsigned int DES_KS[16][2];
#else
typedef unsigned long DES_KS[16][2];
#endif

#if defined(WIN16)
#define DECL int far pascal
#define UCHAR unsigned char far
#define SIZE_T size_t far
#define DES_PKS DES_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define DECL int pascal
#define UCHAR unsigned char
#define SIZE_T size_t
#define DES_PKS DES_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define DECL int
#define UCHAR unsigned char
#define SIZE_T size_t
#define DES_PKS DES_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL des_init(UCHAR *key, int len, DES_KS ks);
DECL des_ecbencode(UCHAR *in64, UCHAR *out64, DES_KS ks);
DECL des_ecbdecode(UCHAR *in64, UCHAR *out64, DES_KS ks);
DECL des_cbcencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks);
DECL des_cbcdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks);
DECL des_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks);
DECL des_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks);
DECL des_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks);
DECL des_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks);
DECL des_3EDEinit(UCHAR *key, int len, DES_KS ks1, DES_KS ks2);
DECL des_ecb3EDEencode(UCHAR *in64, UCHAR *out64, DES_KS ks1, DES_KS ks2);
DECL des_ecb3EDEdecode(UCHAR *in64, UCHAR *out64, DES_KS ks1, DES_KS ks2);
DECL des_cbc3EDEencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_cbc3EDEdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_cfb3EDEencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_cfb3EDEdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_ofb3EDEencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_ofb3EDEdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2);
DECL des_3EEEinit(UCHAR *key, int len, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_ecb3EEEencode(UCHAR *in64, UCHAR *out64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_ecb3EEEdecode(UCHAR *in64, UCHAR *out64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_cbc3EEEencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_cbc3EEEdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_cfb3EEEencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_cfb3EEEdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_ofb3EEEencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);
DECL des_ofb3EEEdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv64, DES_KS ks1, DES_KS ks2, DES_KS ks3);

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* _MODULE_DES3_H_ */
