#ifndef _MODULE_KRYPT_H_
#define _MODULE_KRYPT_H_
/*
 *             libkrypto - Cryptographic Library
 *
 *    Copyright (c) 1998-2015 by CryptoSoft GmbH. All Rights Reserved.
 *              http://www.cryptosoft.de, info@cryptosoft.de
 *
 */

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#if defined(WIN16)
#define DECL int far pascal
#define UCHAR unsigned char far
#define ULONG unsigned long far
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define DECL int pascal
#define UCHAR unsigned char
#define ULONG unsigned long
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define DECL int
#define UCHAR unsigned char
#define ULONG unsigned long
#endif /* !WIN16 && !WIN32 && !OS2 */

typedef DECL (*fn_init)(UCHAR *, int, void *);
typedef DECL (*fn_ecb) (UCHAR *, UCHAR *, void *);
typedef DECL (*fn_cbc) (UCHAR *, UCHAR *, UCHAR *, void *);

/*
 *    Synopsis:	algo_init(key,len,pks)
 * Description: intializes all arrays and permutation tables for algo
 *       Input: pks  -  pointer to key schedule array
 *              key  -  key
 *              len  -  length of key
 *      Return: 0 if OK; >0 if a (semi) weak was selected
 *
 *    Synopsis: algo_ecbYcode(in, out, pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) buffer <in> to
 *              buffer <out> using <algo> in ECB mode
 *     	 Input: in  - pointer to buffer of input data
 *              out - pointer to buffer for output data
 *              pks - pointer to key schedule array
 *      Return: 0 if OK
 *
 *    Synopsis: algo_cbcYcode(in, out, iv, pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) buffer <in> to
 *              to buffer <out> using <algo> in CBC mode
 *     	 Input: in  - pointer to buffer of input data
 *              out - pointer to buffer for output data
 *              iv  - pointer to initialization vector
 *              pks - pointer to key schedule array
 *      Output: 0 if OK
 *
 */

/*
 * Algorithm:  Blowfish 
 */

typedef struct blowfish_schedule
{
  unsigned long P[16 + 2];
  unsigned long S[4*256];
} BLOWFISH_KS;

#if defined(WIN16)
#define BLOWFISH_PKS BLOWFISH_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define BLOWFISH_PKS BLOWFISH_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define BLOWFISH_PKS BLOWFISH_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL blowfish_init(UCHAR *key, int len, BLOWFISH_PKS pks);
DECL blowfish_ecbencode(UCHAR *in64, UCHAR *out64, BLOWFISH_PKS pks);
DECL blowfish_ecbdecode(UCHAR *in64, UCHAR *out64, BLOWFISH_PKS pks);
DECL blowfish_cbcencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, BLOWFISH_PKS pks);
DECL blowfish_cbcdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, BLOWFISH_PKS pks);

/* 
 * Algorithm:   Cast-128
 */

typedef struct cast128_schedule {
  unsigned long ks[32];
} CAST128_KS;

#if defined(WIN16)
#define CAST128_PKS   CAST128_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define CAST128_PKS  CAST128_KS  *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define CAST128_PKS  CAST128_KS  *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL cast128_init(UCHAR *key16, int len, CAST128_PKS pks);
DECL cast128_ecbencode(UCHAR *in64, UCHAR *out64, CAST128_PKS pks);
DECL cast128_ecbdecode(UCHAR *in64, UCHAR *out64, CAST128_PKS pks);
DECL cast128_cbcencode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, CAST128_PKS pks);
DECL cast128_cbcdecode(UCHAR *in64, UCHAR *out64, UCHAR *iv64, CAST128_PKS pks);

/*
 * Algorithm:   DES
 */

typedef unsigned long DES_KS[16][2];

#if defined(WIN16)
#define DES_PKS       DES_KS     far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define DES_PKS      DES_KS      *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define DES_PKS      DES_KS      *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL des_init(UCHAR *k8, int len, DES_KS k);
DECL des_ecbencode(UCHAR *i8, UCHAR *o8, DES_KS k);
DECL des_ecbdecode(UCHAR *i8, UCHAR *o8, DES_KS k);
DECL des_cbcencode(UCHAR *i8, UCHAR *o8, UCHAR *iv8, DES_KS k);
DECL des_cbcdecode(UCHAR *i8, UCHAR *o8, UCHAR *iv8, DES_KS k);
DECL des_3EDEinit(UCHAR *k24, int len, DES_KS k1, DES_KS k2);
DECL des_ecb3EDEencode(UCHAR *i8, UCHAR *o8, DES_KS k1, DES_KS k2);
DECL des_ecb3EDEdecode(UCHAR *i8, UCHAR *o8, DES_KS k1, DES_KS k2);
DECL des_cbc3EDEencode(UCHAR *i8, UCHAR *o8, UCHAR *iv8, DES_KS k1, DES_KS k2);
DECL des_cbc3EDEdecode(UCHAR *i8, UCHAR *o8, UCHAR *iv8, DES_KS k1, DES_KS k2);
DECL des_3EEEinit(UCHAR *k24, int len, DES_KS k1, DES_KS k2, DES_KS k3);
DECL des_ecb3EEEencode(UCHAR *i8, UCHAR *o8, DES_KS k1, DES_KS k2, DES_KS k3);
DECL des_ecb3EEEdecode(UCHAR *i8, UCHAR *o8, DES_KS k1, DES_KS k2, DES_KS k3);
DECL des_cbc3EEEencode(UCHAR *i8, UCHAR *o8, UCHAR *iv8, DES_KS k1, DES_KS k2, DES_KS k3);
DECL des_cbc3EEEdecode(UCHAR *i8, UCHAR *o8, UCHAR *iv8, DES_KS k1, DES_KS k2, DES_KS k3);

/*
 * Algorihtm:   Rijndael
 */

typedef struct rijndael_schedule {
  unsigned long ks[128];
} RIJNDAEL_KS;

#if defined(WIN16)
#define RIJNDAEL_PKS RIJNDAEL_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define RIJNDAEL_PKS RIJNDAEL_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define RIJNDAEL_PKS RIJNDAEL_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL rijndael_init(UCHAR *key32, int len, RIJNDAEL_PKS pks);
DECL rijndael_ecbencode(UCHAR *in128, UCHAR *out128, RIJNDAEL_PKS pks);
DECL rijndael_ecbdecode(UCHAR *in128, UCHAR *out128, RIJNDAEL_PKS pks);
DECL rijndael_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, RIJNDAEL_PKS pks);
DECL rijndael_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, RIJNDAEL_PKS pks);

/*
 * Algorithm:   Safer+ 
 */

typedef struct safer_schedule {
  unsigned char ks[529];
} SAFER_KS;

#if defined(WIN16)
#define SAFER_PKS SAFER_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define SAFER_PKS SAFER_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define SAFER_PKS SAFER_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL safer_init(UCHAR *key32, int len, SAFER_PKS pks);
DECL safer_ecbencode(UCHAR *in128, UCHAR *out128, SAFER_PKS pks);
DECL safer_ecbdecode(UCHAR *in128, UCHAR *out128, SAFER_PKS pks);
DECL safer_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SAFER_PKS pks);
DECL safer_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SAFER_PKS pks);

/*
 * Algorithm:   Serpent
 */

typedef struct serpent_schedule {
  unsigned long ks[140];
} SERPENT_KS;

#if defined(WIN16)
#define SERPENT_PKS SERPENT_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define SERPENT_PKS SERPENT_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define SERPENT_PKS SERPENT_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL serpent_init(UCHAR *key32, int len, SERPENT_PKS pks);
DECL serpent_ecbencode(UCHAR *in128, UCHAR *out128, SERPENT_PKS pks);
DECL serpent_ecbdecode(UCHAR *in128, UCHAR *out128, SERPENT_PKS pks);
DECL serpent_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SERPENT_PKS pks);

/*
 * Algorithm:   Twofish
 */

typedef struct twofish_schedule {
  unsigned long ks[45];
} TWOFISH_KS;

#if defined(WIN16)
#define TWOFISH_PKS TWOFISH_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define TWOFISH_PKS TWOFISH_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define TWOFISH_PKS TWOFISH_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL twofish_init(UCHAR *key32, int len, TWOFISH_PKS pks);
DECL twofish_ecbencode(UCHAR *in128, UCHAR *out128, TWOFISH_PKS pks);
DECL twofish_ecbdecode(UCHAR *in128, UCHAR *out128, TWOFISH_PKS pks);
DECL twofish_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, TWOFISH_PKS pks);

/*
 *   Algorithm: Mars
 */

typedef struct mars_schedule {
  unsigned long ks[40];
} MARS_KS;

#if defined(WIN16)
#define MARS_PKS MARS_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define MARS_PKS MARS_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define MARS_PKS MARS_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */

DECL mars_init(UCHAR *key32, int len, MARS_PKS pks);
DECL mars_ecbencode(UCHAR *in128, UCHAR *out128, MARS_PKS pks);
DECL mars_ecbdecode(UCHAR *in128, UCHAR *out128, MARS_PKS pks);
DECL mars_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, MARS_PKS pks);
DECL mars_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, MARS_PKS pks);

/*
 *  Algorithm: Cast-256
 */

typedef struct cast256_schedule {
  unsigned long ks[96];
} CAST256_KS;                                                                   

#if defined(WIN16)
#define CAST256_PKS CAST256_KS far *
#endif /* WIN16 */
 
#if defined(WIN32) || defined(OS2)
#define CAST256_PKS CAST256_KS *
#endif /* WIN32 */                                                              

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define CAST256_PKS CAST256_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */
                                                                                                        
DECL cast256_init(UCHAR *key32, int len, CAST256_PKS pks);
DECL cast256_ecbencode(UCHAR *in128, UCHAR *out128, CAST256_PKS pks);
DECL cast256_ecbdecode(UCHAR *in128, UCHAR *out128, CAST256_PKS pks);
DECL cast256_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, CAST256_PKS pks);
DECL cast256_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, CAST256_PKS pks);                     

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* _MODULE_KRYPT_H_ */
