/*
 *  Copyright (c) by CryptoSoft GmbH 1998-2015
 *  All Rights Reserved
 *  Licensed Material - Property of CryptoSoft GmbH
 *  This software is made available solely pursuant to the
 *  terms of a license agreement which governs its use.
 * 
 *  TESTVT.C - Variable Text Known Answer Test
 */

#ifdef WIN32
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "krypt.h"	 /* Algorithm function prototypes */

int testvt(char *algo, fn_init pfn_init, fn_ecb pfn_enc, fn_ecb pfn_dec, void *ks, unsigned int keysz, unsigned int blksz)
{
  static unsigned char *kb, *pt, *ct, *rt;
  static unsigned int i, j;

  kb = (unsigned char *) malloc(keysz); memset(kb, 0, keysz);
  pt = (unsigned char *) malloc(blksz); memset(pt, 0, blksz);
  ct = (unsigned char *) malloc(blksz); memset(ct, 0, blksz);
  rt = (unsigned char *) malloc(blksz); memset(rt, 0, blksz);

  if(!kb || !pt || !ct || !rt) {
    fprintf(stderr,"Error: Cannot alloc memory !\n\n");
    return 0;
  }

  printf("Electronic Codebook (ECB) Mode\nVariable Text Known Answer Tests\n");
  printf("Algorithm: %s\nKeysize=%u Bits\n", algo, keysz*8);

  printf("\nKEY=");
  for (j=0; j < keysz; j++) printf("%02x", kb[j]);

  for (i = 0; i < blksz * 8; i++) {
    memset(pt, 0, blksz);
    pt[i/8] = 0x80 >> i%8;
    pfn_init(kb, keysz, ks);
    pfn_enc(pt, ct, ks);
    printf("\n\nI=%u\nPT=", i+1);
    for (j=0; j < blksz; j++) printf("%02x", pt[j]);
    printf("\nCT=");
    for (j=0; j < blksz; j++) printf("%02x", ct[j]);
    pfn_dec(ct, rt, ks);
    if (memcmp(pt,rt,blksz)) {
      fprintf(stderr, "\n*** Decryption for round %d failed ***\n", i);
      fprintf(stderr, "\nKEY=");
      for (j=0; j < keysz; j++) fprintf(stderr, "%02x", kb[j]);
      fprintf(stderr, "\nCT=");
      for (j=0; j < blksz; j++) fprintf(stderr, "%02x", ct[j]);
      fprintf(stderr, "\nPT=");
      for (j=0; j < blksz; j++) fprintf(stderr, "%02x", rt[j]);
      fprintf(stderr, "\nOT=");
      for (j=0; j < blksz; j++) fprintf(stderr, "%02x", pt[j]);
      fprintf(stderr,"\n\nNot Succeeded ...\n\n");
      return 0;
    }
  }
  printf("\n\nSucceeded ...\n\n");
  free(kb); free(pt); free(ct); free(rt);
  return 1;
}

static char       *algo01 = "16 round Blowfish";
static BLOWFISH_KS algo01_ks;
static char       *algo02 = "16 round Cast-128";
static CAST128_KS  algo02_ks;
static char       *algo03 = "16 round DES     ";
static DES_KS      algo03_ks;
static char       *algo04 = "16 round Rijndael";
static RIJNDAEL_KS algo04_ks;
static char       *algo05 = "16 round Safer+  ";
static SAFER_KS    algo05_ks;
static char       *algo06 = "32 round Serpent ";
static SERPENT_KS  algo06_ks;
static char       *algo07 = "16 round Twofish ";
static TWOFISH_KS  algo07_ks;
static char       *algo08 = "16 round Mars    ";
static MARS_KS     algo08_ks;
static char       *algo09 = "48 round Cast-256";
static CAST256_KS  algo09_ks;


int main(void)
{
  testvt(algo01, (fn_init) &blowfish_init, (fn_ecb) &blowfish_ecbencode, (fn_ecb) &blowfish_ecbdecode, &algo01_ks, 56, 8);
  testvt(algo02, (fn_init) &cast128_init, (fn_ecb) &cast128_ecbencode, (fn_ecb) &cast128_ecbdecode, &algo02_ks, 16, 8);
  testvt(algo03, (fn_init) &des_init, (fn_ecb) &des_ecbencode, (fn_ecb) &des_ecbdecode, &(*algo03_ks), 8, 8);
  testvt(algo04, (fn_init) &rijndael_init, (fn_ecb) &rijndael_ecbencode, (fn_ecb) &rijndael_ecbdecode, &algo04_ks, 32, 16);
  testvt(algo05, (fn_init) &safer_init, (fn_ecb) &safer_ecbencode, (fn_ecb) &safer_ecbdecode, &algo05_ks, 32, 16);
  testvt(algo06, (fn_init) &serpent_init, (fn_ecb) &serpent_ecbencode, (fn_ecb) &serpent_ecbdecode, &algo06_ks, 32, 16);
  testvt(algo07, (fn_init) &twofish_init, (fn_ecb) &twofish_ecbencode, (fn_ecb) &twofish_ecbdecode, &algo07_ks, 32, 16);
  testvt(algo08, (fn_init) &mars_init, (fn_ecb) &mars_ecbencode, (fn_ecb) &mars_ecbdecode, &algo08_ks, 32, 16);
  testvt(algo09, (fn_init) &cast256_init, (fn_ecb) &cast256_ecbencode, (fn_ecb) &cast256_ecbdecode, &algo09_ks, 32, 16);
  return 0;
}
