#ifndef _MODULE_SAFER_H_
#define _MODULE_SAFER_H_
/*
 *                     Safer+ - Safer+ Library
 *
 *    Copyright (c) 1998-2017 by CryptoSoft GmbH. All Rights Reserved.
 *              http://www.cryptosoft.de, info@cryptosoft.de
 *
 *    Synopsis:	safer_init(key,len,pks)
 * Description: intializes all arrays and permutation tables for safer
 *       Input: pks  -  pointer to key schedule array
 *              key  -  Safer+ key
 *              len  -  length of key in byte (max. 32)
 *      Output: 0 if OK; >0 if a (semi) weak was selected
 *
 *    Synopsis: safer_ecbYcode(in128,out128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) 128-bit in to
 *              128-bit out using Safer+ in ECB mode
 *     	 Input: in128  - pointer to 128-bit buffer of input data
 *              out128 - pointer to 128-bit buffer for output data
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: safer_cbcYcode(in128,out128,iv128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) 128-bit in to
 *              128-bit out using Safer+ in CBC mode
 *     	 Input: in128  - pointer to 128-bit buffer of input data
 *              out128 - pointer to 128-bit buffer for output data
 *              iv128  - pointer to 128-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: safer_cfbYcode(inpb,outpb,len,iv128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using Safer+ in 8-bit CFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv128 - pointer to 128-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: safer_ofbYcode(inpb,outpb,len,iv128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using Safer+ in 8-bit OFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv128 - pointer to 128-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *
 */


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef struct safer_schedule {
  unsigned char ks[529];
} SAFER_KS;

#if defined(WIN16)
#define DECL int far pascal
#define UCHAR unsigned char far
#define SIZE_T size_t far
#define SAFER_PKS SAFER_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define DECL int pascal
#define UCHAR unsigned char
#define SIZE_T size_t
#define SAFER_PKS SAFER_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define DECL int
#define UCHAR unsigned char
#define SIZE_T size_t 
#define SAFER_PKS SAFER_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */


DECL safer_init(UCHAR *key32, int len, SAFER_PKS pks);
DECL safer_ecbencode(UCHAR *in128, UCHAR *out128, SAFER_PKS pks);
DECL safer_ecbdecode(UCHAR *in128, UCHAR *out128, SAFER_PKS pks);
DECL safer_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SAFER_PKS pks);
DECL safer_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SAFER_PKS pks);
DECL safer_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SAFER_PKS pks);
DECL safer_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SAFER_PKS pks);
DECL safer_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SAFER_PKS pks);
DECL safer_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SAFER_PKS pks);

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* _MODULE_SAFER_H_ */
