#ifndef _MODULE_SERPENT_H_
#define _MODULE_SERPENT_H_
/*
 *                     Serpent - Serpent Library
 *
 *    Copyright (c) 1998-2015 by CryptoSoft GmbH. All Rights Reserved.
 *             http://www.cryptosoft.de, info@cryptosoft.de
 *
 *    Synopsis:	serpent_init(key,len,pks)
 * Description: intializes all arrays and permutation tables for serpent
 *       Input: pks  -  pointer to key schedule array
 *              key  -  Serpent key
 *              len  -  length of key in byte (max. 32)
 *      Output: 0 if OK; >0 if a (semi) weak was selected
 *
 *    Synopsis: serpent_ecbYcode(in128,out128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) 128-bit in to
 *              128-bit out using Serpent in ECB mode
 *     	 Input: in128  - pointer to 128-bit buffer of input data
 *              out128 - pointer to 128-bit buffer for output data
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: serpent_cbcYcode(in128,out128,iv128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) 128-bit in to
 *              128-bit out using Serpent in CBC mode
 *     	 Input: in128  - pointer to 128-bit buffer of input data
 *              out128 - pointer to 128-bit buffer for output data
 *              iv128  - pointer to 128-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: serpent_cfbYcode(inpb,outpb,len,iv128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using Serpent in 8-bit CFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv128 - pointer to 128-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: serpent_ofbYcode(inpb,outpb,len,iv128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using Serpent in 8-bit OFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv128 - pointer to 128-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 */


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef struct serpent_schedule {
  unsigned long ks[140];
} SERPENT_KS;

#if defined(WIN16)
#define DECL int far pascal
#define UCHAR unsigned char far
#define ULONG unsigned long far
#define SIZE_T size_t far
#define SERPENT_PKS SERPENT_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define DECL int pascal
#define UCHAR unsigned char
#define ULONG unsigned long
#define SIZE_T size_t
#define SERPENT_PKS SERPENT_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define DECL int
#define UCHAR unsigned char
#define ULONG unsigned long
#define SIZE_T size_t
#define SERPENT_PKS SERPENT_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */


DECL serpent_init(UCHAR *key32, int len, SERPENT_PKS pks);
DECL serpent_ecbencode(UCHAR *in128, UCHAR *out128, SERPENT_PKS pks);
DECL serpent_ecbdecode(UCHAR *in128, UCHAR *out128, SERPENT_PKS pks);
DECL serpent_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SERPENT_PKS pks);
DECL serpent_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, SERPENT_PKS pks);

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* _MODULE_SERPENT_H_ */
