#ifndef _MODULE_TWOFISH_H_
#define _MODULE_TWOFISH_H_
/*
 *             Twofish - Twofish Library
 *
 *    Copyright (c) 1998-2015 by CryptoSoft GmbH. All Rights Reserved.
 *             http://www.cryptosoft.de, info@cryptosoft.de
 *
 *    Synopsis:	twofish_init(key,len,pks)
 * Description: intializes all arrays and permutation tables for twofish
 *       Input: pks  -  pointer to key schedule array
 *              key  -  Twofish key
 *              len  -  length of key in byte (max. 32)
 *      Output: 0 if OK; >0 if a (semi) weak was selected
 *
 *    Synopsis: twofish_ecbYcode(in128,out128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) 128-bit in to
 *              128-bit out using Twofish in ECB mode
 *     	 Input: in128  - pointer to 128-bit buffer of input data
 *              out128 - pointer to 128-bit buffer for output data
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: twofish_cbcYcode(in128,out128,iv128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) 128-bit in to
 *              128-bit out using Twofish in CBC mode
 *     	 Input: in128  - pointer to 128-bit buffer of input data
 *              out128 - pointer to 128-bit buffer for output data
 *              iv128  - pointer to 128-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: twofish_cfbYcode(inpb,outpb,len,iv128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using Twofish in 8-bit CFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv128 - pointer to 128-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 *    Synopsis: twofish_ofbYcode(inpb,outpb,len,iv128,pks)
 * Description: encrypts (Y=en) or decrypts (Y=de) a given data block
 *              using Twofish in 8-bit OFB mode
 *     	 Input: inpb  - pointer to buffer of input data
 *              outpb - pointer to buffer for output data
 *              len   - length of input buffer AND output buffer
 *              iv128 - pointer to 128-bit initialization vector
 *              pks   - pointer to key schedule array
 *      Output: 0 if OK
 *
 */


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef struct twofish_schedule {
  unsigned long ks[45];
} TWOFISH_KS;

#if defined(WIN16)
#define DECL int far pascal
#define UCHAR unsigned char far
#define ULONG unsigned long far
#define SIZE_T size_t far
#define TWOFISH_PKS TWOFISH_KS far *
#endif /* WIN16 */

#if defined(WIN32) || defined(OS2)
#define DECL int pascal
#define UCHAR unsigned char
#define ULONG unsigned long
#define SIZE_T size_t
#define TWOFISH_PKS TWOFISH_KS *
#endif /* WIN32 */

#if !defined(WIN16) && !defined(WIN32) && !defined(OS2)
#define DECL int
#define UCHAR unsigned char
#define ULONG unsigned long
#define SIZE_T size_t
#define TWOFISH_PKS TWOFISH_KS *
#endif /* !WIN16 && !WIN32 && !OS2 */


DECL twofish_init(UCHAR *key32, int len, TWOFISH_PKS pks);
DECL twofish_ecbencode(UCHAR *in128, UCHAR *out128, TWOFISH_PKS pks);
DECL twofish_ecbdecode(UCHAR *in128, UCHAR *out128, TWOFISH_PKS pks);
DECL twofish_cbcencode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_cbcdecode(UCHAR *in128, UCHAR *out128, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_cfbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_cfbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_ofbencode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, TWOFISH_PKS pks);
DECL twofish_ofbdecode(UCHAR *inpb, UCHAR *outpb, SIZE_T len, UCHAR *iv128, TWOFISH_PKS pks);

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* _MODULE_TWOFISH_H_ */
